package org.example.customer.utility;

/*
 * This is free and unencumbered software released into the public domain.
 * Anyone is free to copy, modify, publish, use, compile, sell, or distribute this software, 
 * either in source code form or as a compiled binary, for any purpose, commercial or 
 * non-commercial, and by any means.
 * 
 * In jurisdictions that recognize copyright laws, the author or authors of this 
 * software dedicate any and all copyright interest in the software to the public domain. 
 * We make this dedication for the benefit of the public at large and to the detriment of 
 * our heirs and successors. We intend this dedication to be an overt act of relinquishment in 
 * perpetuity of all present and future rights to this software under copyright law.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES 
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * For more information, please refer to: https://unlicense.org/
*/


import java.io.Serializable;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.example.websecurity.XssSanitizer;
import org.example.websecurity.XssSanitizerImpl;

/**
 * The Customer Phone for the Customer application.
 <br>
 <br>
 * This class represents the common Phone/Fax field in the Customer Database
 *
 * @author Jonathan Earl
 * @version 1.0
 *
 */
public class Phone implements Serializable
{
    private static final long serialVersionUID = 1L;

    private static final Logger LOG = LogManager.getLogger();

    private String myNumber;

    private XssSanitizer mySanitizer;

    /**
     * The default constructor for the Phone class.
     * <p>
     * The initial values are:
     * <ul>
     * <li>number: null</li>
     * </ul>
     */
    public Phone()
    {
        this(new XssSanitizerImpl());
        LOG.debug("Finishing the default Constructor");
    }

    /**
     * The overloaded constructor for the Phone class that takes an XssSanitizer
     * as input.
     * <p>
     * The initial values are:
     * <ul>
     *   <li>number: null</li>
     * </ul>
     * 
     * @param sanitizer the XssSanitizer used by this instance
     */
    public Phone(final XssSanitizer sanitizer)
    {
        LOG.debug("Starting the overloaded Constructor");
        final String initialPhone = null;

        mySanitizer = sanitizer;

        setNumber(initialPhone);
    }

    /**
     * Returns the number value for the Phone.
     * 
     * @return the number value for the Phone
     */
    public String getNumber()
    {
        LOG.debug("returning the Number: " + myNumber);
        return myNumber;
    }

    /**
     * Sets the number value for the Phone.
     * <p>
     * The business rules are:
     * <ul>
     * <li>the number <strong>may</strong> be null</li>
     * <li>the number must <strong>not</strong> be empty</li>
     * <li>the number must min length of 2 chars</li>
     * <li>the number must max length of 20 chars</li>
     * <li>XSS strings within the number will be removed</li>
     * </ul>
     * 
     * @param number the value to set into the number field
     * @throws IllegalArgumentException if the number is invalid
     */
    public void setNumber(final String number)
    {
        LOG.debug("setting the Number");
        final int max = 30;
        final int min = 2;

        if (number == null)
        {
            LOG.debug("Number is set to null");
            this.myNumber = null;
            return;
        }

        String safeNumber = mySanitizer.sanitizeInput(number);
        if (safeNumber.isEmpty())
        {
            LOG.error("Number must not be empty");
            throw new IllegalArgumentException("Number must not be empty");
        }
        if (safeNumber.length() > max || safeNumber.length() < min)
        {
            LOG.error("Number must be between 2 and 40 chars in length");
            throw new IllegalArgumentException("Number must be between 2 and 20 chars in length");
        }
        LOG.debug("setting the Phone to: " + safeNumber);
        this.myNumber = safeNumber;
    }
    
    /**
     * The hashCode() method of the Phone class.
     * <p>
     * <strong>This method uses:</strong>
     * <ul>
     *  <li>number</li>
     * </ul>
     * 
     * @see java.lang.Object#hashCode()
     * @return the hashCode value for this Phone object
     */
    @Override
    public int hashCode()
    {
        LOG.debug("building HashCode");
        return new HashCodeBuilder()
                .append(getNumber())
                .toHashCode();
    }

    /**
     * The equals() method of the Phone class.
     * <p>
     * <strong>This method uses:</strong>
     * <ul>
     *  <li>number</li>
     * </ul>
     * 
     * @see java.lang.Object#equals(Object obj)
     * @param obj the incoming object to compare against
     * @return true if the fields being compared are equal
     */
    @Override
    public boolean equals(final Object obj)
    {
        LOG.debug("checking equals");
        if (obj instanceof Phone)
        {
            final Phone other = (Phone) obj;
            return new EqualsBuilder()
                    .append(getNumber(), other.getNumber())
                    .isEquals();
        }
        else
        {
            return false;
        }
    }

    /**
     * The toString method for the Phone class.
     * 
     * this method will return:<br>
     * Phone [Number=XXX]
     */
    @Override
    public String toString()
    {
        return "Phone [Number=" + myNumber + "]";
    }

}
