package org.example.customer;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.when;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.utility.Location;
import org.example.customer.utility.Phone;
import org.example.websecurity.XssSanitizer;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;

public class CustomerTest 
{
    private Customer testCustomer = null;
    private XssSanitizer sanitizerMock = null; 

    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Customer.class).getName(), Level.WARN);
    }

    @Before
    public void setUp() throws Exception
    {
        sanitizerMock = Mockito.mock(XssSanitizer.class);
        // tell the Mock Sanitizer to return the first argument it was passed
        // as a trimmed String
        when(sanitizerMock.sanitizeInput(anyString())).thenAnswer(i -> ((String) i.getArguments()[0]).trim());

        testCustomer = new Customer(sanitizerMock);
    }

    @Test
    public void testCustomerDefaultConstructor()
    {
        Customer defaultCustomer = new Customer();
        assertNotNull(defaultCustomer);
    }

    @Test
    public void testDefaultFirstName()
    {
        String expected = "John";
        String actual = testCustomer.getFirstName();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullFirstName()
    {
        testCustomer.setFirstName(null);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyFirstName()
    {
        testCustomer.setFirstName("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongFirstName()
    {
        String bad = RandomStringUtils.randomAlphabetic(41);
        testCustomer.setFirstName(bad);
    }
    
    @Test
    public void testValidFirstName()
    {
        String[] names = {"Barney", "Fred", "betty", "WILMA"};
        for (String name : names)
        {
            testCustomer.setFirstName(name);
        }
    }

    @Test
    public void testSanitizedFirstName()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Jonathan";
        String xssSanitized = "Jonathan";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testCustomer.setFirstName(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "Jonathan";
        String actual = testCustomer.getFirstName();
        assertEquals(expected, actual);
    }

    @Test
    public void testDefaultLastName()
    {
        String expected = "Smith";
        String actual = testCustomer.getLastName();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullLastName()
    {
        testCustomer.setLastName(null);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyLastName()
    {
        testCustomer.setLastName("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongLastName()
    {
        String bad = RandomStringUtils.randomAlphabetic(41);
        testCustomer.setLastName(bad);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testTooShortLastName()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testCustomer.setLastName(bad);
    }
    
    @Test
    public void testValidLastName()
    {
        String[] names = {"Rubble", "Flintstone"};
        for (String name : names)
        {
            testCustomer.setLastName(name);
        }
    }

    @Test
    public void testSanitizedLastName()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Earl";
        String xssSanitized = "Earl";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testCustomer.setLastName(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "Earl";
        String actual = testCustomer.getLastName();
        assertEquals(expected, actual);
    }
    
    @Test 
    public void testDefaultLocation()
    {
        Location expected = new Location();
        Location actual = testCustomer.getLocation();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullLocation()
    {
        testCustomer.setLocation(null);
    }
    
    @Test 
    public void testValidLocation()
    {
        Location home = new Location();
        home.setCity("Vancouver");
        home.setCountry("Canada");
        testCustomer.setLocation(home);
    }
    
    @Test 
    public void testDefaultPhone()
    {
        Phone expected = new Phone();
        Phone actual = testCustomer.getPhone();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testNullPhone()
    {
        testCustomer.setPhone(null);
    }

    @Test
    public void testValidPhone()
    {
        Phone phone = new Phone();
        phone.setNumber("001 42-234-1234");
        testCustomer.setPhone(phone);
        assertNotNull(testCustomer.getPhone());
    }

    @Test  
    public void testToString()
    {
        String expected = "Customer [Id=1, FirstName=John, "
                + "LastName=Smith, City=New York City, "
                + "Country=United States of America, Phone=null]";
        String actual = testCustomer.toString();
        assertEquals(expected, actual);
    }
    
    @Test public void testHashCode()
    {
        Customer sample = new Customer();
        int expected = sample.hashCode();
        int actual = testCustomer.hashCode();
        assertEquals(expected, actual);
    }
    
    @Test public void testEquals()
    {
        Customer sample = new Customer();
        assertTrue(testCustomer.equals(sample));
    }
    
    @Test public void testNotEquals()
    {
        Customer sample = new Customer();
        sample.setLastName("Jones");
        assertFalse(testCustomer.equals(sample));
        
        Object junk = new Object();
        assertFalse(testCustomer.equals(junk));
    }
}
