package org.example.customer;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.websecurity.XssSanitizer;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;

public class OrderTest
{
    private Order testOrder = null;
    private XssSanitizer sanitizerMock = null;

    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Order.class).getName(), Level.WARN);
    }

    @Before
    public void setUp() throws Exception
    {
        sanitizerMock = mock(XssSanitizer.class);
        // tell the Mock Sanitizer to return the first argument it was passed
        // as a trimmed String
        when(sanitizerMock.sanitizeInput(anyString())).thenAnswer(i -> ((String) i.getArguments()[0]).trim());

        testOrder = new Order(sanitizerMock);
    }

    @Test
    public void testOrderDefaultConstructor()
    {
        Order defaultOrder = new Order();
        assertNotNull(defaultOrder);
    }

    @Test
    public void testDefaultOrderDate()
    {
        LocalDate expected = LocalDate.now();
        LocalDate actual = testOrder.getOrderDate();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullOrderDate()
    {
        testOrder.setOrderDate(null);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testFutureOrderDate()
    {
        LocalDate bad = LocalDate.now().plusDays(1);
        testOrder.setOrderDate(bad);
    }
    
    @Test
    public void testYesterdayOrderDate()
    {
        LocalDate good = LocalDate.now().minusDays(100);
        testOrder.setOrderDate(good);
    }
    
    @Test
    public void testDefaultCustomerId()
    {
        int expected = 1;
        int actual = testOrder.getCustomerId();
        assertEquals(expected, actual);
    }

    @Test
    public void testValidCustomerId()
    {
        int[] good = { 2, 34, 999, 22, 1, 565656 };
        for (int entry : good) {
            testOrder.setCustomerId(entry);
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void testInValidCustomerId()
    {
        testOrder.setCustomerId(0);
    }
    
    @Test
    public void testDefaultTotalAmount()
    {
        double expected = 0.00;
        double actual = testOrder.getTotalAmount();
        double precision = 0.001;
        assertEquals(expected, actual, precision);
    }

    @Test
    public void testValidTotalAmount()
    {
        double[] good = {2.0, 0.34, 999.50, 22.25, 1.0};
        for (double entry : good) {
            testOrder.setTotalAmount(entry);
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void testInValidTotalAmount()
    {
        testOrder.setTotalAmount(0.0);
    }
    
    @Test
    public void testDefaultOrderItems()
    {
        int expected = 0;
        int actual = testOrder.getOrderItems().size();
        assertEquals(expected, actual);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testNullItems()
    {
        testOrder.setOrderItems(null);
    }
    
    @Test
    public void testOrderItemsTotalAmount()
    {
        List<OrderItem> mockItems = new ArrayList<OrderItem>();
        
        OrderItem orderItem1Mock = mock(OrderItem.class);
        when(orderItem1Mock.getSubTotal()).thenReturn(12.50);
        mockItems.add(orderItem1Mock);
        
        OrderItem orderItem2Mock = Mockito.mock(OrderItem.class);
        when(orderItem2Mock.getSubTotal()).thenReturn(1.75);
       mockItems.add(orderItem2Mock);
        
        OrderItem orderItem3Mock = Mockito.mock(OrderItem.class);
        when(orderItem3Mock.getSubTotal()).thenReturn(34.10);
        mockItems.add(orderItem3Mock);
        
        testOrder.setOrderItems(mockItems);
        
        double expected = 48.35;
        double actual = testOrder.getTotalAmount();
        double precision = 0.001;
        assertEquals(expected, actual, precision);
    }

    @Test
    public void testDefaultOrderNumber()
    {
        assertNull(testOrder.getOrderNumber());
    }

    @Test
    public void testNullOrderNumber()
    {
        testOrder.setOrderNumber(null);
    }

    @Test(expected = IllegalArgumentException.class)
    public void OrderNumber()
    {
        testOrder.setOrderNumber("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongOrderNumber()
    {
        String bad = RandomStringUtils.randomAlphabetic(11);
        testOrder.setOrderNumber(bad);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testTooShortOrderNumber()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testOrder.setOrderNumber(bad);
    }
    
    @Test
    public void testValidOrderNumber()
    {
        String[] values = {"ABC-011", "2022-12X"};
        for (String value : values)
        {
            testOrder.setOrderNumber(value);
        }
    }

    @Test
    public void testSanitizedOrderNumber()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Ord23 W";
        String xssSanitized = "Ord23 W";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testOrder.setOrderNumber(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "Ord23 W";
        String actual = testOrder.getOrderNumber();
        assertEquals(expected, actual);
        
        verify(sanitizerMock, times(1)).sanitizeInput(xssInput);
    }
    
    @Test  
    public void testToString()
    {
        LocalDate today = LocalDate.now();
        String formattedToday = today.toString();
        String expected = "Order [Id=1, OrderDate="
                + formattedToday + ", CustomerId=1, "
                + "TotalAmount=0.0, OrderNumber=null, "
                + "OrderItems=0]";
        String actual = testOrder.toString();
        assertEquals(expected, actual);
    }
    
    @Test public void testHashCode()
    {
        Order sample = new Order();
        int expected = sample.hashCode();
        int actual = testOrder.hashCode();
        assertEquals(expected, actual);
    }
    
    @Test public void testEquals()
    {
        Order sample = new Order();
        assertTrue(testOrder.equals(sample));
    }
    
    @Test public void testNotEquals()
    {
        Order sample = new Order();
        sample.setCustomerId(2);
        assertFalse(testOrder.equals(sample));
        
        Customer junk = new Customer();
        assertFalse(testOrder.equals(junk));
    }
}
