package org.example.customer;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.when;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.utility.CustomerEntity;
import org.example.websecurity.XssSanitizer;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;

public class ProductTest
{
    private Product testProduct = null;
    private XssSanitizer sanitizerMock = null;

    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Product.class).getName(), Level.WARN);
    }

    @Before
    public void setUp() throws Exception
    {
        sanitizerMock = Mockito.mock(XssSanitizer.class);
        // tell the Mock Sanitizer to return the first argument it was passed
        // as a trimmed String
        when(sanitizerMock.sanitizeInput(anyString())).thenAnswer(i -> ((String) i.getArguments()[0]).trim());

        testProduct = new Product(sanitizerMock);
    }

    @Test
    public void testCustomerDefaultConstructor()
    {
        Product defaultProduct = new Product();
        assertNotNull(defaultProduct);
    }

    @Test
    public void testDefaultProductName()
    {
        String expected = "Paper Envelope";
        String actual = testProduct.getProductName();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullProductName()
    {
        testProduct.setProductName(null);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyProductName()
    {
        testProduct.setProductName("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongProductName()
    {
        String bad = RandomStringUtils.randomAlphabetic(51);
        testProduct.setProductName(bad);
    }

    @Test
    public void testValidProductName()
    {
        String[] names = {"Copy Paper", "Flour", "Sugar", "10in Frying Pan"};
        for (String name : names)
        {
            testProduct.setProductName(name);
        }
    }

    @Test
    public void testSanitizedProductName()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Frozen Strawberries";
        String xssSanitized = "Frozen Strawberries";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testProduct.setProductName(xssInput);

        // now we will test to see if the Product class calls the sanitizer correctly
        String expected = "Frozen Strawberries";
        String actual = testProduct.getProductName();
        assertEquals(expected, actual);
    }

    @Test
    public void testDefaultSupplierId()
    {
        int expected = 1;
        int actual = testProduct.getSupplierId();
        assertEquals(expected, actual);
    }

    @Test
    public void testValidSupplierId()
    {
        int[] good = { 2, 34, 999, 22, 1, 565656 };
        for (int entry : good) {
            testProduct.setSupplierId(entry);
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void testInValidSupplierId()
    {
        testProduct.setSupplierId(0);
    }

    @Test
    public void testDefaultUnitPrice()
    {
        double expected = 0.01;
        double actual = testProduct.getUnitPrice();
        double precision = 0.001;
        assertEquals(expected, actual, precision);
    }

    @Test
    public void testValidUnitPrice()
    {
        double[] good = {2.0, 0.34, 999.50, 22.25, 1.0};
        for (double entry : good) {
            testProduct.setUnitPrice(entry);
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void testInValidUnitPrice()
    {
        testProduct.setUnitPrice(0.0);
    }

    @Test
    public void testDefaultPackaging()
    {
        String expected = "single";
        String actual = testProduct.getPackaging();
        assertEquals(expected, actual);
    }

    @Test
    public void testNullPackaging()
    {
        testProduct.setPackaging(null);
        assertNull(testProduct.getPackaging());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testWhitespacePackaging()
    {
        testProduct.setPackaging("  ");
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testEmptyPackaging()
    {
        testProduct.setPackaging("");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongPackaging()
    {
        String bad = RandomStringUtils.randomAlphabetic(31);
        testProduct.setPackaging(bad);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooShortPackaging()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testProduct.setPackaging(bad);
    }

    @Test
    public void testValidPackaging()
    {
        String[] values = {"ream", "dozen", "5 count"};
        for (String value : values)
        {
            testProduct.setPackaging(value);
        }
    }

    @Test
    public void testSanitizedPackaging()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Case ";
        String xssSanitized = "Case";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testProduct.setPackaging(xssInput);

        // now we will test to see if the Product class calls the sanitizer correctly
        String expected = "Case";
        String actual = testProduct.getPackaging();
        assertEquals(expected, actual);
    }

    @Test
    public void testDefaultDiscontinued()
    {
        assertFalse(testProduct.isDiscontinued());
    }

    @Test
    public void testSettingDiscontinued()
    {
        testProduct.setDiscontinued(true);
        assertTrue(testProduct.isDiscontinued());
    }

    @Test
    public void testToString()
    {
        String expected = "Product [Id=1, "
                + "ProductName=Paper Envelope, "
                + "SupplierId=1, UnitPrice=0.01, "
                + "Packaging=single, Discontinued=false]";
        String actual = testProduct.toString();
        assertEquals(expected, actual);
    }

    @Test public void testHashCode()
    {
        Product sample = new Product();
        int expected = sample.hashCode();
        int actual = testProduct.hashCode();
        assertEquals(expected, actual);
    }

    @Test public void testEquals()
    {
        Product sample = new Product();
        assertTrue(testProduct.equals(sample));
    }

    @Test public void testNotEquals()
    {
        Product sample = new Product();
        sample.setSupplierId(2);
        assertFalse(testProduct.equals(sample));

        CustomerEntity junk = new Customer();
        assertFalse(testProduct.equals(junk));
    }

}
