package org.example.customer;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.utility.Location;
import org.example.customer.utility.Phone;
import org.example.websecurity.XssSanitizer;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;

public class SupplierTest
{
    private Supplier testSupplier = null;
    private XssSanitizer sanitizerMock = null; 

    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Supplier.class).getName(), Level.WARN);
    }

    @Before
    public void setUp() throws Exception
    {
        sanitizerMock = Mockito.mock(XssSanitizer.class);
        // tell the Mock Sanitizer to return the first argument it was passed
        // as a trimmed String
        when(sanitizerMock.sanitizeInput(anyString())).thenAnswer(i -> ((String) i.getArguments()[0]).trim());

        testSupplier = new Supplier(sanitizerMock);
    }
    
    @Test
    public void testSupplierDefaultConstructor()
    {
        Supplier defaultSupplier = new Supplier();
        assertNotNull(defaultSupplier);
    }
    
    @Test
    public void testDefaultCompanyName()
    {
        String expected = "ABC Inc.";
        String actual = testSupplier.getCompanyName();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullCompanyName()
    {
        testSupplier.setCompanyName(null);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyCompanyName()
    {
        testSupplier.setCompanyName("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongCompanyName()
    {
        String bad = RandomStringUtils.randomAlphabetic(41);
        testSupplier.setCompanyName(bad);
    }
    
    @Test
    public void testValidCompanyName()
    {
        String[] names = {"Players Unlimited", "Will Work for Food"};
        for (String name : names)
        {
            testSupplier.setCompanyName(name);
        }
    }

    @Test
    public void testSanitizedCompanyName()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>StarShip Enterprises";
        String xssSanitized = "StarShip Enterprises";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testSupplier.setCompanyName(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "StarShip Enterprises";
        String actual = testSupplier.getCompanyName();
        assertEquals(expected, actual);
        
        verify(sanitizerMock, times(1)).sanitizeInput(xssInput);
    }
    
    @Test
    public void testDefaultContactName()
    {
        String expected = "John Smith";
        String actual = testSupplier.getContactName();
        assertEquals(expected, actual);
    }

    @Test
    public void testNullContactName()
    {
        testSupplier.setContactName(null);
        assertNull(testSupplier.getContactName());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyContactName()
    {
        testSupplier.setContactName("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongContactName()
    {
        String bad = RandomStringUtils.randomAlphabetic(51);
        testSupplier.setContactName(bad);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testTooShortContactName()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testSupplier.setContactName(bad);
    }
    
    @Test
    public void testValidContactName()
    {
        String[] names = {"Betty Rubble", "Sir Robert Service"};
        for (String name : names)
        {
            testSupplier.setContactName(name);
        }
    }

    @Test
    public void testSanitizedContactName()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Bill Norton";
        String xssSanitized = "Bill Norton";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testSupplier.setContactName(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "Bill Norton";
        String actual = testSupplier.getContactName();
        assertEquals(expected, actual);
        
        verify(sanitizerMock, times(1)).sanitizeInput(xssInput);
    }
    
    @Test 
    public void testDefaultLocation()
    {
        Location expected = new Location();
        Location actual = testSupplier.getLocation();
        assertEquals(expected, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void testNullLocation()
    {
        testSupplier.setLocation(null);
    }
    
    @Test 
    public void testValidLocation()
    {
        Location home = new Location();
        home.setCity("Vancouver");
        home.setCountry("Canada");
        testSupplier.setLocation(home);
    }
    
    @Test 
    public void testDefaultPhone()
    {
        Phone expected = new Phone();
        Phone actual = testSupplier.getPhone();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testNullPhone()
    {
        testSupplier.setPhone(null);
    }

    @Test
    public void testValidPhone()
    {
        Phone phone = new Phone();
        phone.setNumber("001 42-234-1234");
        testSupplier.setPhone(phone);
        assertNotNull(testSupplier.getPhone());
    }
    
    @Test 
    public void testDefaultFax()
    {
        Phone expected = new Phone();
        Phone actual = testSupplier.getFax();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testNullFax()
    {
        testSupplier.setFax(null);
    }

    @Test
    public void testValidFax()
    {
        Phone phone = new Phone();
        phone.setNumber("001 42-234-1234");
        testSupplier.setFax(phone);
        assertNotNull(testSupplier.getFax());
    }
    
    @Test  
    public void testToString()
    {
        String expected = "Supplier [Id=1, CompanyName=ABC Inc., "
                + "City=New York City, Country=United States of America, "
                + "Phone=null]";
        String actual = testSupplier.toString();
        assertEquals(expected, actual);
    }
    
    @Test public void testHashCode()
    {
        Supplier sample = new Supplier();
        int expected = sample.hashCode();
        int actual = testSupplier.hashCode();
        assertEquals(expected, actual);
    }
    
    @Test public void testEquals()
    {
        Supplier sample = new Supplier();
        assertTrue(testSupplier.equals(sample));
    }
    
    @Test public void testNotEquals()
    {
        Supplier sample = new Supplier();
        sample.setCompanyName("World Wide Unlimited");
        assertFalse(testSupplier.equals(sample));
        
        Customer junk = new Customer();
        assertFalse(testSupplier.equals(junk));
    }

}
