package org.example.customer.utility;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.when;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.websecurity.XssSanitizer;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;

public class LocationTest
{
    private Location testLocation = null;
    private XssSanitizer sanitizerMock = null;
    
    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Location.class).getName(), Level.WARN);
    }

    @Before
    public void setUp() throws Exception
    {
        sanitizerMock = Mockito.mock(XssSanitizer.class);
        // tell the Mock Sanitizer to return the first argument it was passed
        // as a trimmed String
        when(sanitizerMock.sanitizeInput(anyString())).thenAnswer(i -> ((String) i.getArguments()[0]).trim());

        testLocation = new Location(sanitizerMock);
    }
    
    @Test
    public void testLocationDefaultConstructor()
    {
        Location defaultLocation = new Location();
        assertNotNull(defaultLocation);
    }
    
    @Test
    public void testDefaultCity()
    {
        String expected = "New York City";
        String actual = testLocation.getCity();
        assertEquals(expected, actual);
    }

    @Test
    public void testNullCity()
    {
        testLocation.setCity(null);
        assertNull(testLocation.getCity());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyCity()
    {
        testLocation.setCity("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongCity()
    {
        String bad = RandomStringUtils.randomAlphabetic(41);
        testLocation.setCity(bad);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testTooShortCity()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testLocation.setCity(bad);
    }
    
    @Test
    public void testValidCity()
    {
        String[] names = {"Boulder", "London"};
        for (String name : names)
        {
            testLocation.setCity(name);
        }
    }

    @Test
    public void testSanitizedCity()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>Paris ";
        String xssSanitized = "Paris";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testLocation.setCity(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "Paris";
        String actual = testLocation.getCity();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testDefaultCountry()
    {
        String expected = "United States of America";
        String actual = testLocation.getCountry();
        assertEquals(expected, actual);
    }

    @Test
    public void testNullCountry()
    {
        testLocation.setCountry(null);
        assertNull(testLocation.getCountry());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyCountry()
    {
        testLocation.setCountry("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongCountry()
    {
        String bad = RandomStringUtils.randomAlphabetic(41);
        testLocation.setCountry(bad);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testTooShortCountry()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testLocation.setCountry(bad);
    }
    
    @Test
    public void testValidCountry()
    {
        String[] names = {"Canada", "Japan", "Brazil"};
        for (String name : names)
        {
            testLocation.setCountry(name);
        }
    }
    
    @Test  
    public void testToString()
    {
        String expected = "Location [City=New York City, Country=United States of America]";
        String actual = testLocation.toString();
        assertEquals(expected, actual);
    }
    
    @Test public void testHashCode()
    {
        Location sample = new Location();
        int expected = sample.hashCode();
        int actual = testLocation.hashCode();
        assertEquals(expected, actual);
    }
    
    @Test public void testEquals()
    {
        Location sample = new Location();
        assertTrue(testLocation.equals(sample));
    }
    
    @Test public void testNotEquals()
    {
        Location sample = new Location();
        sample.setCity("Boston");
        assertFalse(testLocation.equals(sample));
        
        Object junk = new Object();
        assertFalse(testLocation.equals(junk));
    }

}
