package org.example.customer.utility;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.when;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.websecurity.XssSanitizer;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mockito;

public class PhoneTest
{
    
    private Phone testPhone = null;
    private XssSanitizer sanitizerMock = null;
    
    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Phone.class).getName(), Level.WARN);
    }

    @Before
    public void setUp() throws Exception
    {
        sanitizerMock = Mockito.mock(XssSanitizer.class);
        // tell the Mock Sanitizer to return the first argument it was passed
        // as a trimmed String
        when(sanitizerMock.sanitizeInput(anyString())).thenAnswer(i -> ((String) i.getArguments()[0]).trim());

        testPhone = new Phone(sanitizerMock);
    }
    
    @Test
    public void testPhoneDefaultConstructor()
    {
        Phone defaultLocation = new Phone();
        assertNotNull(defaultLocation);
    }
    
    @Test
    public void testDefaultNumber()
    {
        assertNull(testPhone.getNumber());
    }

    @Test
    public void testNullNumber()
    {
        testPhone.setNumber(null);
        assertNull(testPhone.getNumber());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testEmptyNumber()
    {
        testPhone.setNumber("  ");
    }

    @Test(expected = IllegalArgumentException.class)
    public void testTooLongNumber()
    {
        String bad = RandomStringUtils.randomAlphabetic(31);
        testPhone.setNumber(bad);
    }
    
    @Test(expected = IllegalArgumentException.class)
    public void testTooShortNumber()
    {
        String bad = RandomStringUtils.randomAlphabetic(1);
        testPhone.setNumber(bad);
    }
    
    @Test
    public void testValidNumber()
    {
        String[] values = {"212.555.1212", "(303)555-1212"};
        for (String value : values)
        {
            testPhone.setNumber(value);
        }
    }
    
    @Test
    public void testSanitizedNumber()
    {
        String xssInput = "<script>alert('You have been Hacked!');</script>202.555.1212 ";
        String xssSanitized = "202.555.1212";
        // tell the Mock Sanitizer to return a specific string
        when(sanitizerMock.sanitizeInput(xssInput)).thenReturn(xssSanitized);

        testPhone.setNumber(xssInput);

        // now we will test to see if the Customer class calls the sanitizer correctly
        String expected = "202.555.1212";
        String actual = testPhone.getNumber();
        assertEquals(expected, actual);
    }

    @Test  
    public void testToString()
    {
        String expected = "Phone [Number=null]";
        String actual = testPhone.toString();
        assertEquals(expected, actual);
    }
    
    @Test public void testHashCode()
    {
        Phone sample = new Phone();
        int expected = sample.hashCode();
        int actual = testPhone.hashCode();
        assertEquals(expected, actual);
    }
    
    @Test public void testEquals()
    {
        Phone sample = new Phone();
        sample.setNumber("202.555.1212");
        testPhone.setNumber("202.555.1212");
        assertTrue(testPhone.equals(sample));
    }
    
    @Test public void testNotEquals()
    {
        Phone sample = new Phone();
        sample.setNumber("312.101.3200");
        assertFalse(testPhone.equals(sample));
        
        Object junk = new Object();
        assertFalse(testPhone.equals(junk));
    }

}
