package org.example.customerdao;

/*
* This is free and unencumbered software released into the public domain.
* Anyone is free to copy, modify, publish, use, compile, sell, or distribute this software, 
* either in source code form or as a compiled binary, for any purpose, commercial or 
* non-commercial, and by any means.
* 
* In jurisdictions that recognize copyright laws, the author or authors of this 
* software dedicate any and all copyright interest in the software to the public domain. 
* We make this dedication for the benefit of the public at large and to the detriment of 
* our heirs and successors. We intend this dedication to be an overt act of relinquishment in 
* perpetuity of all present and future rights to this software under copyright law.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
* INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
* PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES 
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
* 
* For more information, please refer to: https://unlicense.org/
*/

import java.util.List;

import javax.sql.DataSource;

import org.example.customer.Product;
import org.example.customer.Supplier;
import org.example.customerdao.utility.Deleteable;
import org.example.websecurity.UserCredentials;


/**
 * This is the ProductDAO Interface for the Customer DAO component of the Customer Web Application.
 * This will be the primary Product database exposure for the Customer Web Layer.
 * 
 * @author Jonathan Earl
 * @since 1.0
 * 
 */
public interface ProductDAO extends Deleteable
{
    /**
     * Sets the Read Only DataSource for this DAO Implication.
     * 
     * @param readOnlyDS the readOnlyDS to set
     */
    void setReadOnlyDS(DataSource readOnlyDS);
    
    /**
     * Sets the Read/Write DataSource for this DAO Implication.
     * 
     * @param readWriteDS the readWriteDS to set
     */
    void setReadWriteDS(DataSource readWriteDS);
    
    /**
     * Finds all Products.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @return a list of Products, this list may be empty
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    List<Product> findAllProducts(UserCredentials credentials);   
    
    /**
     * Finds all non discontinued Products.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @return a list of active Products, this list may be empty
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    List<Product> findAllActiveProducts(UserCredentials credentials);   
    
    /**
     * Find Product by Id.
     * <p>
     * This will return null if the Product cannot be found.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param id the Product ID to search for
     * @return a Product or null if the Product is not found
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    Product findProductById(UserCredentials credentials, int id);
    
    /**
     * Find Products by Supplier.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param supplier the Supplier to search for
     * @return a list of Products, this list may be empty
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    List<Product> findProductsBySupplier(UserCredentials credentials, Supplier supplier);
    
    /**
     * Adds a Product.
     * <p >
     * Only a Manager is permitted to add a Product.
     * 
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param product the Product object to add
     * @return the id of the new Product record
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    int addProduct(UserCredentials credentials, Product product);
    
    /**
     * Updates a Product.
     * <p >
     * Only a Manager is permitted to update a Product.
     * 
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param product the Product object to update
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    void updateProduct(UserCredentials credentials, Product product);
    
}
