package org.example.customerdao;

/*
 * This is free and unencumbered software released into the public domain.
 * Anyone is free to copy, modify, publish, use, compile, sell, or distribute this software, 
 * either in source code form or as a compiled binary, for any purpose, commercial or 
 * non-commercial, and by any means.
 * 
 * In jurisdictions that recognize copyright laws, the author or authors of this 
 * software dedicate any and all copyright interest in the software to the public domain. 
 * We make this dedication for the benefit of the public at large and to the detriment of 
 * our heirs and successors. We intend this dedication to be an overt act of relinquishment in 
 * perpetuity of all present and future rights to this software under copyright law.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES 
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * For more information, please refer to: https://unlicense.org/
*/

import java.util.List;

import javax.sql.DataSource;

import org.example.customer.Supplier;
import org.example.customerdao.utility.Deleteable;
import org.example.websecurity.UserCredentials;

/**
 * This is the SupplierDAO Interface for the Customer DAO component of the Customer Web Application.
 * This will be the primary Supplier database exposure for the Customer Web Layer.
 * 
 * @author Jonathan Earl
 * @since 1.0
 * 
 */
public interface SupplierDAO extends Deleteable
{
    /**
     * Sets the Read Only DataSource for this DAO Implication.
     * 
     * @param readOnlyDS the readOnlyDS to set
     */
    void setReadOnlyDS(DataSource readOnlyDS);
    
    /**
     * Sets the Read/Write DataSource for this DAO Implication.
     * 
     * @param readWriteDS the readWriteDS to set
     */
    void setReadWriteDS(DataSource readWriteDS);
    
    /**
     * Finds all Suppliers.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @return a list of Suppliers, this list may be empty
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    List<Supplier> findAllSuppliers(UserCredentials credentials);   
    
    /**
     * Find Supplier by Id.
     * <p>
     * This will return null if the Supplier cannot be found.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param id the Supplier ID to search for
     * @return a Supplier or null if the Supplier is not found
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    Supplier findSupplierById(UserCredentials credentials, int id);
    
    /**
     * Find Suppliers by Company name.
     * <p>
     * This will use a Wild Card search to find any Supplier with the provided string
     * and a portion of their Company name.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param name the string to search for
     * @return a list if Suppliers, this list may be empty
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    List<Supplier> findSuppliersByCompanyName(UserCredentials credentials, String name);
    
    /**
     * Adds a Supplier.
     * <p >
     * Only a Manager is permitted to add a Supplier.
     * 
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param supplier the Supplier object to add
     * @return the id of the new Supplier record
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    int addSupplier(UserCredentials credentials, Supplier supplier);
    
    /**
     * Updates a Supplier.
     * <p >
     * Only a Manager is permitted to update a Supplier.
     * 
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param supplier the Supplier object to update
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    void updateSupplier(UserCredentials credentials, Supplier supplier);
}
