package org.example.customerdao.utility;

import org.example.customer.utility.CustomerEntity;

/*
* This is free and unencumbered software released into the public domain.
* Anyone is free to copy, modify, publish, use, compile, sell, or distribute this software, 
* either in source code form or as a compiled binary, for any purpose, commercial or 
* non-commercial, and by any means.
* 
* In jurisdictions that recognize copyright laws, the author or authors of this 
* software dedicate any and all copyright interest in the software to the public domain. 
* We make this dedication for the benefit of the public at large and to the detriment of 
* our heirs and successors. We intend this dedication to be an overt act of relinquishment in 
* perpetuity of all present and future rights to this software under copyright law.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
* INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
* PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES 
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
* 
* For more information, please refer to: https://unlicense.org/
*/

import org.example.websecurity.UserCredentials;

/**
 * This is the Deletable interface for the DAO components of the
 * Customer Web Application. 
 * 
 * @author Jonathan Earl
 * @since 1.0
 * 
 */
public interface Deleteable
{
    /**
     * This method will determine if a CustomerEntity record can be deleted.
     * <p>
     * Only <strong>Managers</strong> can delete CustomerEntity records.
     * 
     * @param credentials the UserCredentials for this Database operation 
     * @param id the if of the object to delete
     * @return true if the object is deleteable
     */
    boolean isDeleteable(UserCredentials credentials, int id);
    
    /**
     * Checks to see if a Product has Orders, if not the Product is deleteable.
     * <p>
     * Only <strong>Managers</strong> can delete CustomerEntity records.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param entity the CustomerEntity object to check
     * @return true  if the CustomerEntity is deleteable
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     */
    boolean isDeleteable(UserCredentials credentials, CustomerEntity entity);
    
    /**
     * Deletes an CustomerEntity.
     * <p >
     * Only a Manager is permitted to delete a CustomerEntity.
     * <p>
     * If this CustomerEntity fails to meet the criteria for deletion, 
     * the delete will fail. 
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param entity the CustomerEntity object to delete
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     * @throws NonDeleteableRecordException if this CustomerEntity cannot be deleted
     */
    void deleteEntity(UserCredentials credentials, CustomerEntity entity)
       throws NonDeleteableRecordException;
    
    /**
     * Deletes a CuUstomerEntity by id.
     * <p >
     * Only a Manager is permitted to delete a CustomerEntity.
     * 
     * @param credentials the UserCredentials for this Database operation
     * @param id the CustomerEntity id to delete
     * @throws IllegalArgumentException 
     *              if there is a connection or permission problem
     * @throws NonDeleteableRecordException if this CustomerEntity cannot be deleted
     */
    void deleteEntity(UserCredentials credentials, int id)
       throws NonDeleteableRecordException;
}
