package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.Customer;
import org.example.customer.utility.CustomerEntity;
import org.example.customer.utility.Location;
import org.example.customer.utility.Phone;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.customerdao.utility.NonDeleteableRecordException;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class CustomerDAOFastTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
      
    private static DBUnitJDBCUtility utility = null;
    
    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Customer.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }   
    
    @AfterClass
    public static void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindAllCustomers() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
           
        handler.findAllCustomers(null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindAllCustomers() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        handler.findAllCustomers(userCredentialsMock);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindAllCustomers() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
    
        List<Customer> records = handler.findAllCustomers(userCredentialsMock);
        assertNotNull(records);
        assertTrue(records.size() == 91);

    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindCustomerById() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        int id = 1;
    
        handler.findCustomerById(null, id);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindCustomerById() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
        
        int id = 1;
    
        handler.findCustomerById(userCredentialsMock, id);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindCustomerById() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        int id = 1;
    
        Customer record = handler.findCustomerById(userCredentialsMock, id);
        assertNotNull(record);
        String expected = "Anders";
        String actual = record.getLastName();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testFindCustomerByInvalidId() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        int id = 101;
    
        Customer record = handler.findCustomerById(userCredentialsMock, id);
        assertNull(record);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindCustomersByName() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
      
        String firstName ="John";
        String lastName = "Steel";
    
        handler.findCustomersByName(null, firstName, lastName);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindCustomersByName() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        String firstName ="John";
        String lastName = "Steel";
    
        handler.findCustomersByName(userCredentialsMock, firstName, lastName);
    }
    
    @Test
    public void testFindCustomersByName() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        String firstName ="John";
        String lastName = "Steel";
    
        List<Customer> records = handler.findCustomersByName(userCredentialsMock, firstName, lastName);
        assertNotNull(records);
        assertTrue(records.size() == 1);
        assertTrue(records.get(0).getId() == 43);
    }
    
    @Test
    public void testFindNoCustomersByName() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        String firstName ="Fred";
        String lastName = "Flintstone";
    
        List<Customer> records = handler.findCustomersByName(userCredentialsMock, firstName, lastName);
        assertNotNull(records);
        assertTrue(records.size() == 0);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsAddCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
          Customer mockCustomer = buildMockCustomer();
        
        handler.addCustomer(null, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedAddCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Customer mockCustomer = buildMockCustomer();
        
        handler.addCustomer(userCredentialsMock, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullAddCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.addCustomer(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsUpdateCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
          
        Customer mockCustomer = buildMockCustomer();
        
        handler.updateCustomer(null, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedUpdateCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Customer mockCustomer = buildMockCustomer();
        
        handler.updateCustomer(userCredentialsMock, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullUpdateCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.updateCustomer(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
                
        Customer mockCustomer = buildMockCustomer();
        
        handler.isDeleteable(null, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
        
        Customer mockCustomer = buildMockCustomer();
        
        handler.isDeleteable(userCredentialsMock, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullIsDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test
    public void testIsDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        Customer mockCustomer = buildMockCustomer();
        // Customer 71 has no Orders
        mockCustomer.setId(71);
         
        assertTrue(handler.isDeleteable(userCredentialsMock, mockCustomer));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidEntityDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        CustomerEntity mockCustomerEntity = new CustomerEntity();
         
        handler.deleteEntity(userCredentialsMock, mockCustomerEntity);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidEntityIsDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        CustomerEntity mockCustomerEntity = new CustomerEntity();
         
        assertTrue(handler.isDeleteable(userCredentialsMock, mockCustomerEntity));
    }
    
    @Test
    public void testIsNotDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        Customer mockCustomer = buildMockCustomer();
        // Customer 70 has Orders
        mockCustomer.setId(70);
        
        assertFalse(handler.isDeleteable(userCredentialsMock, mockCustomer));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableId() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        handler.isDeleteable(null, 21);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableId() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
              
        handler.isDeleteable(userCredentialsMock, 21);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullIsDeleteableId() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testIsDeleteableId() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        // Customer 71 has no Orders
        assertTrue(handler.isDeleteable(userCredentialsMock, 71));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testIsNotDeleteableId() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
                  
        assertFalse(handler.isDeleteable(userCredentialsMock, 70));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsDeleteCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        Customer mockCustomer = buildMockCustomer();
        
        handler.deleteEntity(null, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedDeleteCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Customer mockCustomer = buildMockCustomer();
        
        handler.deleteEntity(userCredentialsMock, mockCustomer);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullDeleteCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.deleteEntity(userCredentialsMock, null);
    }
    
    @Test(expected = NonDeleteableRecordException.class)  
    public void testNotDeleteableCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Customer mockCustomer = buildMockCustomer();
        // Customer 70 has Orders
        mockCustomer.setId(70);
        
        handler.deleteEntity(userCredentialsMock, mockCustomer);
    }
   
    private Customer buildMockCustomer()
    {
        Customer mockCustomer = new Customer();
        // leave the id as default, the database will generate it
        mockCustomer.setFirstName("Robert");
        mockCustomer.setLastName("Service");
        Location mockLocation = new Location();
        mockLocation.setCity("Dawson City");
        mockLocation.setCountry("Canada");
        mockCustomer.setLocation(mockLocation);
        Phone mockPhone = new Phone();
        mockPhone.setNumber("(867)993-5566");
        mockCustomer.setPhone(mockPhone);
        
        return mockCustomer;
    }
}
