package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.dbunit.Assertion;
import org.dbunit.dataset.ITable;
import org.example.customer.Customer;
import org.example.customer.utility.Location;
import org.example.customer.utility.Phone;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class CustomerDAOSlowTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
    private static final String CUSTOMER_ADD_FILE = configDir + File.separator + "addCustomer.xml";
    private static final String CUSTOMER_UPDATE_FILE = configDir + File.separator + "updateCustomer.xml";
    private static final String CUSTOMER_DELETE_FILE = configDir + File.separator + "deleteCustomer.xml";
    
    private static DBUnitJDBCUtility utility = null;
    
    @Before  // (re)build the Database before each test
    public void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Customer.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @After
    public void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Category(SmokeTest.class)
    @Test(timeout = 3000) // 3 sec (3000) millisec timeout
    public void testAddCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Customer mockCustomer = buildMockCustomer();
        
        int expected = 92;
        int actual = handler.addCustomer(userCredentialsMock, mockCustomer);
        assertEquals(expected, actual);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("CUSTOMER");
            ITable expectedTable = utility.getTableFromFile(CUSTOMER_ADD_FILE, "CUSTOMER");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
    
    @Test(timeout = 1000) 
    public void testUpdateCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Customer mockCustomer68 = handler.findCustomerById(userCredentialsMock, 68);
        mockCustomer68.setFirstName("Penny");
        mockCustomer68.setLastName("Miller");
        
        handler.updateCustomer(userCredentialsMock, mockCustomer68);
        
        Customer mockCustomer8 = handler.findCustomerById(userCredentialsMock, 8);
        mockCustomer8.getLocation().setCity("Barcelona");
        
        handler.updateCustomer(userCredentialsMock, mockCustomer8);
        
        Customer mockCustomer25 = handler.findCustomerById(userCredentialsMock, 25);
        mockCustomer25.setPhone(null);
        
        handler.updateCustomer(userCredentialsMock, mockCustomer25);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("CUSTOMER");
            ITable expectedTable = utility.getTableFromFile(CUSTOMER_UPDATE_FILE, "CUSTOMER");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
      
    @Test(timeout = 1000) // 1 sec (1000) millisec timeout
    public void testDeleteCustomer() 
            throws Exception
    {
        CustomerDAO handler = new CustomerDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Customer mockCustomer = buildMockCustomer();
        // Customer 71 has no Orders
        mockCustomer.setId(71);
         
        handler.deleteEntity(userCredentialsMock, mockCustomer);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("CUSTOMER");
            ITable expectedTable = utility.getTableFromFile(CUSTOMER_DELETE_FILE, "CUSTOMER");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }

    private Customer buildMockCustomer()
    {        
        Customer mockCustomer = new Customer();
        // leave the id as default, the database will generate it
        mockCustomer.setFirstName("Robert");
        mockCustomer.setLastName("Service");
        Location mockLocation = new Location();
        mockLocation.setCity("Dawson City");
        mockLocation.setCountry("Canada");
        mockCustomer.setLocation(mockLocation);
        Phone mockPhone = new Phone();
        mockPhone.setNumber("(867)993-5566");
        mockCustomer.setPhone(mockPhone);
        
        return mockCustomer;
    }
}
