package org.example.customerdao;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import static org.mockito.Mockito.when;

import java.io.File;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.Customer;
import org.example.customer.Order;
import org.example.customer.OrderItem;
import org.example.customer.utility.CustomerEntity;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class OrderDAOFastTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
      
    private static DBUnitJDBCUtility utility = null;
    
    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Order.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @AfterClass
    public static void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindAllOrders() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
           
        handler.findAllOrders(null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindAllOrders() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        handler.findAllOrders(userCredentialsMock);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindAllOrders() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
    
        List<Order> records = handler.findAllOrders(userCredentialsMock);
        assertNotNull(records);
        assertTrue(records.size() == 830);
        Order order323 = records.get(322);
        // <ORDERS ID="323" ORDER_DATE="2013-06-17" CUSTOMER_ID="51" TOTAL_AMOUNT="2595.00" ORDER_NUMBER="542700"/>
        // <ORDER_ITEM ID="861" ORDER_ID="323" PRODUCT_ID="11" UNIT_PRICE="21.00" QUANTITY="15"/>
        // <ORDER_ITEM ID="862" ORDER_ID="323" PRODUCT_ID="56" UNIT_PRICE="38.00" QUANTITY="60"/>
       
        String expectedOrderNumber = "542700";
        String acutalOrderNumber = order323.getOrderNumber();
        assertEquals(expectedOrderNumber, acutalOrderNumber);
        
        List<OrderItem> order323OrderItems = order323.getOrderItems();
        
        int expectedOrderItemCount = 2;
        int actualOrderItemCount = order323OrderItems.size();
        assertEquals(expectedOrderItemCount, actualOrderItemCount);
        
        double expectedSubTotalCost = 21.00 * 15;
        double acutalSubTotalCost = order323OrderItems.get(0).getSubTotal();
        double precision = 0.001;
        assertEquals(expectedSubTotalCost, acutalSubTotalCost, precision);

    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindAllOrderItems() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        Order mockOrder = buildMockOrder();
           
        handler.findAllOrderItems(null, mockOrder);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindAllOrderItemss() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        Order mockOrder = buildMockOrder();
        
        handler.findAllOrderItems(userCredentialsMock, mockOrder);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullOrderFindAllOrderItems() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        handler.findAllOrderItems(userCredentialsMock, null);
    }
    
    @Test
    public void testFindAllOrderItems() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Order mockOrder = buildMockOrder();
        
        List<OrderItem> records = handler.findAllOrderItems(userCredentialsMock, mockOrder);
        
       // <ORDER_ITEM ID="56" ORDER_ID="21" PRODUCT_ID="29" UNIT_PRICE="99.00" QUANTITY="10"/>
       // <ORDER_ITEM ID="57" ORDER_ID="21" PRODUCT_ID="72" UNIT_PRICE="27.80" QUANTITY="4"/>

        assertNotNull(records);
        assertTrue(records.size() == 2);
        OrderItem orderItem56 = records.get(0);
          
        int expectedProductId = 29;
        int acutalProductId = orderItem56.getProductId();
        assertEquals(expectedProductId, acutalProductId);
        
        double expectedSubTotalCost = 99.00 * 10;
        double acutalSubTotalCost = orderItem56.getSubTotal();
        double precision = 0.001;
        assertEquals(expectedSubTotalCost, acutalSubTotalCost, precision);

    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindOrderById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
           
        handler.findOrderById(null, 1);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindOrderById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        handler.findOrderById(userCredentialsMock, 1);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindOrderById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Order order323 = handler.findOrderById(userCredentialsMock, 323);
    
        // <ORDERS ID="323" ORDER_DATE="2013-06-17" CUSTOMER_ID="51" TOTAL_AMOUNT="2595.00" ORDER_NUMBER="542700"/>
        // <ORDER_ITEM ID="861" ORDER_ID="323" PRODUCT_ID="11" UNIT_PRICE="21.00" QUANTITY="15"/>
        // <ORDER_ITEM ID="862" ORDER_ID="323" PRODUCT_ID="56" UNIT_PRICE="38.00" QUANTITY="60"/>
       
        String expectedOrderNumber = "542700";
        String acutalOrderNumber = order323.getOrderNumber();
        assertEquals(expectedOrderNumber, acutalOrderNumber);
        
        List<OrderItem> order323OrderItems = order323.getOrderItems();
        
        int expectedOrderItemCount = 2;
        int actualOrderItemCount = order323OrderItems.size();
        assertEquals(expectedOrderItemCount, actualOrderItemCount);
        
        double expectedSubTotalCost = 21.00 * 15;
        double acutalSubTotalCost = order323OrderItems.get(0).getSubTotal();
        double precision = 0.001;
        assertEquals(expectedSubTotalCost, acutalSubTotalCost, precision);

    }
    
    @Test
    public void testInvalidFindOrderById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        assertNull(handler.findOrderById(userCredentialsMock, 921));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindOrderByCustomer() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        Customer mockCustomer = new Customer();
        mockCustomer.setId(6);
           
        handler.findOrdersByCustomer(null, mockCustomer);
    }
    
   
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindOrdersByCustomer() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        Customer mockCustomer = new Customer();
        mockCustomer.setId(6);
           
        handler.findOrdersByCustomer(userCredentialsMock, mockCustomer);
    }
    
   
    @Test(expected = IllegalArgumentException.class)  
    public void testNullFindOrdersByCustomer() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
           
        handler.findOrdersByCustomer(userCredentialsMock, null);
    }
    
    @Test
    public void testFindOrdersByCustomer() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Customer mockCustomer = new Customer();
        mockCustomer.setId(6);
           
        List<Order> records = handler.findOrdersByCustomer(userCredentialsMock, mockCustomer);
        
        //   <ORDERS ID="254" ORDER_DATE="2013-04-09" CUSTOMER_ID="6" TOTAL_AMOUNT="149.00" ORDER_NUMBER="542631"/>
        //   <ORDERS ID="262" ORDER_DATE="2013-04-17" CUSTOMER_ID="6" TOTAL_AMOUNT="136.80" ORDER_NUMBER="542639"/>
        //   <ORDERS ID="335" ORDER_DATE="2013-06-27" CUSTOMER_ID="6" TOTAL_AMOUNT="330.00" ORDER_NUMBER="542712"/>
        //   <ORDERS ID="367" ORDER_DATE="2013-07-29" CUSTOMER_ID="6" TOTAL_AMOUNT="464.00" ORDER_NUMBER="542744"/>
        //   <ORDERS ID="606" ORDER_DATE="2014-01-27" CUSTOMER_ID="6" TOTAL_AMOUNT="625.00" ORDER_NUMBER="542983"/>
        //   <ORDERS ID="709" ORDER_DATE="2014-03-17" CUSTOMER_ID="6" TOTAL_AMOUNT="677.00" ORDER_NUMBER="543086"/>
        //   <ORDERS ID="811" ORDER_DATE="2014-04-29" CUSTOMER_ID="6" TOTAL_AMOUNT="858.00" ORDER_NUMBER="543188"/>
       
        assertNotNull(records);
        assertTrue(records.size() == 7);
        Order order367 = records.get(3);
          
        LocalDate expectedOrderDate = LocalDate.of(2013, 7, 29);
        LocalDate acutalOrderDate = order367.getOrderDate();
        assertEquals(expectedOrderDate, acutalOrderDate);
        
        double expectedTotalAmount = 464.00;
        double acutalTotalAmount = order367.getTotalAmount();
        double precision = 0.001;
        assertEquals(expectedTotalAmount, acutalTotalAmount, precision);

    }
 
    @Test
    public void testInvalidFindOrdersByCustomer() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Customer mockCustomer = new Customer();
        mockCustomer.setId(930);
           
        List<Order> records = handler.findOrdersByCustomer(userCredentialsMock, mockCustomer);
               
        assertNotNull(records);
        assertTrue(records.size() == 0);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsDeleteOrderById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
       handler.deleteEntity(null, 1);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsDeleteOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        Order mockOrder = buildMockOrder();
      
        handler.deleteEntity(null, mockOrder);
    }
   
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedDeleteOrderById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
    
        handler.deleteEntity(userCredentialsMock, 1);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedDeleteOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
    
        Order mockOrder = buildMockOrder();
        
        handler.deleteEntity(userCredentialsMock, mockOrder);
    }
    
   
    @Test(expected = IllegalArgumentException.class)  
    public void testNullOrderDeleteOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
           
        handler.deleteEntity(userCredentialsMock, null);
    }
   
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
    
        handler.isDeleteable(userCredentialsMock, 1);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteable() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
    
        Order mockOrder = buildMockOrder();
        handler.isDeleteable(userCredentialsMock, mockOrder);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
          
        handler.isDeleteable(null, 1); 
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteable() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
          
        Order mockOrder = buildMockOrder();
        handler.isDeleteable(null, mockOrder);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullOrderIsDeleteable() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
    
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testWrongEntityIsDeleteable() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        CustomerEntity entity = new CustomerEntity();
    
        handler.isDeleteable(userCredentialsMock, entity);
    }
    
    @Test  
    public void testIsDeleteableById() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        assertTrue(handler.isDeleteable(userCredentialsMock, 1));
    }
    
    @Test  
    public void testIsDeleteable() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Order mockOrder = buildMockOrder();
    
        assertTrue(handler.isDeleteable(userCredentialsMock, mockOrder));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsAddOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        Order mockOrder = buildMockOrder();
      
        handler.addOrder(null, mockOrder);
    }
    
   
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedAddOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        Order mockOrder = buildMockOrder();
        
        handler.addOrder(userCredentialsMock, mockOrder);
    }
    
   
    @Test(expected = IllegalArgumentException.class)  
    public void testNullOrderAddOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
           
        handler.addOrder(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedUpdateOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
    
        Order mockOrder = buildMockOrder();
        
        handler.updateOrder(userCredentialsMock, mockOrder);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsUpdateOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
          
        Order mockOrder = buildMockOrder();
        
        handler.updateOrder(null, mockOrder);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullOrderUpdateOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
          
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
       
        handler.updateOrder(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidOrderUpdateOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
    
        Order mockOrder = buildMockOrder();
        mockOrder.setId(1299);
        
        handler.updateOrder(userCredentialsMock, mockOrder);
    }
    
    private Order buildMockOrder()
    { 
        Order mockOrder = new Order();
        mockOrder.setId(21);
        mockOrder.setOrderDate(LocalDate.now());
        mockOrder.setCustomerId(7);
        mockOrder.setTotalAmount(20.50);
        mockOrder.setOrderNumber("0292929");
        List<OrderItem> mockOrderItems = buildOrderItems();
        mockOrder.setOrderItems(mockOrderItems);
        
        return mockOrder;
    }
    
    
    private List<OrderItem> buildOrderItems()
    {
        List<OrderItem> data = new ArrayList<OrderItem>();
        for (int counter = 1; counter < 3; counter++)
        {
            OrderItem mockOrderItem = new OrderItem();
            mockOrderItem.setId(counter);
            mockOrderItem.setOrderId(21);
            mockOrderItem.setProductId(12);
            mockOrderItem.setUnitPrice(10.25);
            mockOrderItem.setQuantity(2);
            data.add(mockOrderItem);
        }
        
        return data;
    }
}
