package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.dbunit.Assertion;
import org.dbunit.dataset.ITable;
import org.example.customer.Order;
import org.example.customer.OrderItem;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class OrderDAOSlowTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
    private static final String Order_ADD_FILE = configDir + File.separator + "addOrder.xml";
    private static final String Order_UPDATE_FILE = configDir + File.separator + "updateOrder.xml";
    private static final String Order_DELETE_FILE = configDir + File.separator + "deleteOrder.xml";
    
    private static DBUnitJDBCUtility utility = null;
    
    @Before  // (re)build the Database before each test
    public void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Order.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @After
    public void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Category(SmokeTest.class)
    @Test(timeout = 3000)
    public void testAddOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Order mockOrder = buildMockOrder();
        
        int expected = 831;
        int actual = handler.addOrder(userCredentialsMock, mockOrder);
        assertEquals(expected, actual);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("Orders");
            ITable expectedTable = utility.getTableFromFile(Order_ADD_FILE, "Orders");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
    
    @Test(timeout = 1000) 
    public void testUpdateOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Order mockOrder140 = handler.findOrderById(userCredentialsMock, 140);
        // Original data
        // <ORDERS ID="140" ORDER_DATE="2012-12-18" CUSTOMER_ID="70" TOTAL_AMOUNT="1058.40" ORDER_NUMBER="542517"/>
        // <ORDER_ITEM ID="368" ORDER_ID="140" PRODUCT_ID="24" UNIT_PRICE="3.60" QUANTITY="15"/>  
        // <ORDER_ITEM ID="369" ORDER_ID="140" PRODUCT_ID="28" UNIT_PRICE="36.40" QUANTITY="6"/>  
        // <ORDER_ITEM ID="370" ORDER_ID="140" PRODUCT_ID="59" UNIT_PRICE="44.00" QUANTITY="12"/>  
        // <ORDER_ITEM ID="371" ORDER_ID="140" PRODUCT_ID="71" UNIT_PRICE="17.20" QUANTITY="15"/>  
       
        mockOrder140.setTotalAmount(1097.40);
        
        // Modified data
        // <ORDERS ID="140" ORDER_DATE="2012-12-18" CUSTOMER_ID="70" TOTAL_AMOUNT="1097.40" ORDER_NUMBER="542517"/>
        // <ORDER_ITEM ID="2156" ORDER_ID="140" PRODUCT_ID="24" UNIT_PRICE="3.60" QUANTITY="15"/> 54.00
        // <ORDER_ITEM ID="2157" ORDER_ID="140" PRODUCT_ID="28" UNIT_PRICE="36.40" QUANTITY="6"/> 218.40
        // <ORDER_ITEM ID="2158" ORDER_ID="140" PRODUCT_ID="59" UNIT_PRICE="44.00" QUANTITY="12"/> 528.00
        // <ORDER_ITEM ID="2159" ORDER_ID="140" PRODUCT_ID="71" UNIT_PRICE="17.20" QUANTITY="15"/> 258.00
        // <ORDER_ITEM ID="2160" ORDER_ID="140" PRODUCT_ID="17" UNIT_PRICE="39.00" QUANTITY="1"/> 258.00
        
        List<OrderItem> mockOrder140items = mockOrder140.getOrderItems();
        OrderItem mockOrderItem = new OrderItem();
        mockOrderItem.setOrderId(140);
        mockOrderItem.setProductId(17);
        mockOrderItem.setUnitPrice(39.00);
        mockOrderItem.setQuantity(1);
        mockOrder140items.add(mockOrderItem);
        
        handler.updateOrder(userCredentialsMock, mockOrder140);

        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("Orders");
            ITable expectedTable = utility.getTableFromFile(Order_UPDATE_FILE, "Orders");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
      
    @Test(timeout = 3000) 
    public void testDeleteOrder() 
            throws Exception
    {
        OrderDAO handler = new OrderDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Order mockOrder = buildMockOrder();
        mockOrder.setId(117);
         
        handler.deleteEntity(userCredentialsMock, mockOrder);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("Orders");
            ITable expectedTable = utility.getTableFromFile(Order_DELETE_FILE, "Orders");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }

    private Order buildMockOrder()
    { 
        // <ORDERS ID="831" ORDER_DATE="2020-02-17" CUSTOMER_ID="12" TOTAL_AMOUNT="266.00" ORDER_NUMBER="632901"/>
        
        Order mockOrder = new Order();
        // leave id blank the DB will generate it
        mockOrder.setOrderDate(LocalDate.of(2020, 2, 17));
        mockOrder.setCustomerId(12);
        mockOrder.setTotalAmount(266.00);
        mockOrder.setOrderNumber("632901");
        List<OrderItem> mockOrderItems = buildOrderItems();
        mockOrder.setOrderItems(mockOrderItems);
        
        return mockOrder;
    }
    
    
    private List<OrderItem> buildOrderItems()
    {
     //   <ORDER_ITEM ID="2156" ORDER_ID="831" PRODUCT_ID="3" UNIT_PRICE="10.00" QUANTITY="2"/>
     //   <ORDER_ITEM ID="2157" ORDER_ID="831" PRODUCT_ID="52" UNIT_PRICE="7.00" QUANTITY="12"/>
     //   <ORDER_ITEM ID="2158" ORDER_ID="831" PRODUCT_ID="762" UNIT_PRICE="18.00" QUANTITY="9"/>
        
        List<OrderItem> data = new ArrayList<OrderItem>();
        
        OrderItem mockOrderItem1 = new OrderItem();
        mockOrderItem1.setOrderId(831);
        mockOrderItem1.setProductId(3);
        mockOrderItem1.setUnitPrice(10.00);
        mockOrderItem1.setQuantity(2);
        data.add(mockOrderItem1);
        
        OrderItem mockOrderItem2 = new OrderItem();
        mockOrderItem2.setOrderId(831);
        mockOrderItem2.setProductId(52);
        mockOrderItem2.setUnitPrice(7.00);
        mockOrderItem2.setQuantity(12);
        data.add(mockOrderItem2);
        
        OrderItem mockOrderItem3 = new OrderItem();
        mockOrderItem3.setOrderId(831);
        mockOrderItem3.setProductId(762);
        mockOrderItem3.setUnitPrice(18.00);
        mockOrderItem3.setQuantity(9);
        data.add(mockOrderItem3);
       
        return data;
    }
}
