package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.Product;
import org.example.customer.Supplier;
import org.example.customer.utility.CustomerEntity;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.customerdao.utility.NonDeleteableRecordException;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class ProductDAOFastTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
      
    private static DBUnitJDBCUtility utility = null;
     
    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Product.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @AfterClass
    public static void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindAllProducts() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
           
        handler.findAllProducts(null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindAllProducts() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        handler.findAllProducts(userCredentialsMock);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindAllProducts() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
    
        List<Product> records = handler.findAllProducts(userCredentialsMock);
        assertNotNull(records);
        assertTrue(records.size() == 80);

    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindAllActiveProducts() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
           
        handler.findAllActiveProducts(null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindAllActiveProducts() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        handler.findAllActiveProducts(userCredentialsMock);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindAllActiveProducts() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
    
        List<Product> records = handler.findAllActiveProducts(userCredentialsMock);
        assertNotNull(records);
        assertTrue(records.size() == 71);

    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindProductById() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        int id = 1;
    
        handler.findProductById(null, id);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindProductById() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
        
        int id = 1;
    
        handler.findProductById(userCredentialsMock, id);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindProductByValidId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        int id = 1;
    
        Product record = handler.findProductById(userCredentialsMock, id);
        assertNotNull(record);
        String expected = "Chai";
        String actual = record.getProductName();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testFindProductByInvalidId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        int id = 101;
    
        Product record = handler.findProductById(userCredentialsMock, id);
        assertNull(record);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindProductsBySupplier() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
      
        Supplier supplier = new Supplier();
        supplier.setId(10);
    
        handler.findProductsBySupplier(null, supplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindProductsBySupplier() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        Supplier supplier = new Supplier();
        supplier.setId(10);
    
        handler.findProductsBySupplier(userCredentialsMock, supplier);
    }
    
    @Test
    public void testFindProductsBySupplier() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Supplier supplier = new Supplier();
        supplier.setId(10);
    
        List<Product> records = handler.findProductsBySupplier(userCredentialsMock, supplier);
        assertNotNull(records);
        assertTrue(records.size() == 1);
        assertTrue(records.get(0).getId() == 24);
    }
    
    @Test
    public void testFindNoProductsBySupplier() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        Supplier supplier = new Supplier();
        supplier.setId(83);
    
        List<Product> records = handler.findProductsBySupplier(userCredentialsMock, supplier);
        assertNotNull(records);
        assertTrue(records.size() == 0);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsAddProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        Product mockProduct = buildMockProduct();
        
        handler.addProduct(null, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedAddProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Product mockProduct = buildMockProduct();
        
        handler.addProduct(userCredentialsMock, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullAddProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.addProduct(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullProductUpdateProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
       
       handler.updateProduct(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsUpdateProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
          
        Product mockProduct = buildMockProduct();
        
        handler.updateProduct(null, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedUpdateProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Product mockProduct = buildMockProduct();
        
        handler.updateProduct(userCredentialsMock, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullUpdateProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.updateProduct(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
                
        Product mockProduct = buildMockProduct();
        
        handler.isDeleteable(null, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
        
        Product mockProduct = buildMockProduct();
        
        handler.isDeleteable(userCredentialsMock, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullIsDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test
    public void testIsDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        Product mockProduct = buildMockProduct();
        // Product 17 has no Orders
        mockProduct.setId(17);
         
        assertTrue(handler.isDeleteable(userCredentialsMock, mockProduct));
    }
    
    @Test
    public void testIsNotDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        Product mockProduct = buildMockProduct();
        // Product 12 has Orders
        mockProduct.setId(12);
        
        assertFalse(handler.isDeleteable(userCredentialsMock, mockProduct));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        handler.isDeleteable(null, 21);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
              
        handler.isDeleteable(userCredentialsMock, 21);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullIsDeleteableId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testIsDeleteableId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        // Product 17 has no Orders
        assertTrue(handler.isDeleteable(userCredentialsMock, 17));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testIsNotDeleteableId() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
                  
        assertFalse(handler.isDeleteable(userCredentialsMock, 14));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsDeleteProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        Product mockProduct = buildMockProduct();
        
        handler.deleteEntity(null, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedDeleteProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Product mockProduct = buildMockProduct();
        
        handler.deleteEntity(userCredentialsMock, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullDeleteProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.deleteEntity(userCredentialsMock, null);
    }
    
    @Test(expected = NonDeleteableRecordException.class)  
    public void testNotDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Product mockProduct = buildMockProduct();
        // Product 12 has Orders
        mockProduct.setId(12);
        
        handler.deleteEntity(userCredentialsMock, mockProduct);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidEntityDeleteableProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        CustomerEntity mockCustomerEntity = new CustomerEntity();
         
        handler.deleteEntity(userCredentialsMock, mockCustomerEntity);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidEntityIsDeleteableCustomer() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        CustomerEntity mockCustomerEntity = new CustomerEntity();
         
        assertTrue(handler.isDeleteable(userCredentialsMock, mockCustomerEntity));
    }
   
    private Product buildMockProduct()
    { 
        Product mockProduct = new Product();
        // leave the id as default, the database will generate it
        mockProduct.setProductName("Salted Peanuts");
        mockProduct.setSupplierId(7);
        mockProduct.setUnitPrice(0.50);
        mockProduct.setPackaging("1 kg pkg");
        mockProduct.setDiscontinued(false);
        
        return mockProduct;
    }
}
