package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.dbunit.Assertion;
import org.dbunit.dataset.ITable;
import org.example.customer.Product;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class ProductDAOSlowTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
    private static final String Product_ADD_FILE = configDir + File.separator + "addProduct.xml";
    private static final String Product_UPDATE_FILE = configDir + File.separator + "updateProduct.xml";
    private static final String Product_DELETE_FILE = configDir + File.separator + "deleteProduct.xml";
    
    private static DBUnitJDBCUtility utility = null;
    
    @Before  // (re)build the Database before each test
    public void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Product.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @After
    public void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Category(SmokeTest.class)
    @Test(timeout = 3000) // 3 sec (3000) millisec timeout
    public void testAddProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Product mockProduct = buildMockProduct();
        
        int expected = 81;
        int actual = handler.addProduct(userCredentialsMock, mockProduct);
        assertEquals(expected, actual);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("Product");
            ITable expectedTable = utility.getTableFromFile(Product_ADD_FILE, "Product");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
    
    @Test(timeout = 1000) 
    public void testUpdateProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Product mockProduct14 = handler.findProductById(userCredentialsMock, 14);
        mockProduct14.setProductName("Firm Tofu");
        
        handler.updateProduct(userCredentialsMock, mockProduct14);
        
        Product mockProduct55 = handler.findProductById(userCredentialsMock, 55);
        mockProduct55.setDiscontinued(true);
        
        handler.updateProduct(userCredentialsMock, mockProduct55);
        
        Product mockProduct71 = handler.findProductById(userCredentialsMock, 71);
        mockProduct71.setUnitPrice(23.75);
        
        handler.updateProduct(userCredentialsMock, mockProduct71);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("Product");
            ITable expectedTable = utility.getTableFromFile(Product_UPDATE_FILE, "Product");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
      
    @Test(timeout = 1000) // 1 sec (1000) millisec timeout
    public void testDeleteProduct() 
            throws Exception
    {
        ProductDAO handler = new ProductDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Product mockProduct = buildMockProduct();
        // Product 17 has no orders
        mockProduct.setId(17);
         
        handler.deleteEntity(userCredentialsMock, mockProduct);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("Product");
            ITable expectedTable = utility.getTableFromFile(Product_DELETE_FILE, "Product");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }

    private Product buildMockProduct()
    {   
        Product mockProduct = new Product();
        // leave the id as default, the database will generate it
        mockProduct.setProductName("Salted Peanuts");
        mockProduct.setSupplierId(7);
        mockProduct.setUnitPrice(0.50);
        mockProduct.setPackaging("1 kg pkg");
        mockProduct.setDiscontinued(false);
        
        return mockProduct;
    }
}
