package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.example.customer.Supplier;
import org.example.customer.utility.CustomerEntity;
import org.example.customer.utility.Location;
import org.example.customer.utility.Phone;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.customerdao.utility.NonDeleteableRecordException;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class SupplierDAOFastTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
      
    private static DBUnitJDBCUtility utility = null;
    
    @BeforeClass // this runs only once before any test
    public static void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Supplier.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @AfterClass
    public static void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindAllSuppliers() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
           
        handler.findAllSuppliers(null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindAllSuppliers() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        handler.findAllSuppliers(userCredentialsMock);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindAllSuppliers() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
    
        List<Supplier> records = handler.findAllSuppliers(userCredentialsMock);
        assertNotNull(records);
        assertTrue(records.size() == 29);

    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindSupplierById() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        int id = 1;
    
        handler.findSupplierById(null, id);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindSupplierById() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
        
        int id = 1;
    
        handler.findSupplierById(userCredentialsMock, id);
    }
    
    @Category(SmokeTest.class)
    @Test
    public void testFindSupplierByValidId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        int id = 1;
    
        Supplier record = handler.findSupplierById(userCredentialsMock, id);
        assertNotNull(record);
        String expected = "Exotic Liquids";
        String actual = record.getCompanyName();
        assertEquals(expected, actual);
    }
    
    @Test
    public void testFindSupplierByInvalidId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        int id = 101;
    
        Supplier record = handler.findSupplierById(userCredentialsMock, id);
        assertNull(record);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsFindSuppliersByName() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
      
        String company ="John";
    
        handler.findSuppliersByCompanyName(null, company);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedFindSuppliersByName() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
    
        String company ="John";
        
        handler.findSuppliersByCompanyName(userCredentialsMock, company);
    }
    
    @Test
    public void testFindSuppliersByName() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        String company ="Trad";
    
        List<Supplier> records = handler.findSuppliersByCompanyName(userCredentialsMock,company);
        assertNotNull(records);
        assertTrue(records.size() == 2);
        assertTrue(records.get(0).getId() == 4);
    }
    
    @Test
    public void testFindNoSuppliersByName() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        
        String company ="John";
    
        List<Supplier> records = handler.findSuppliersByCompanyName(userCredentialsMock, company);
        assertNotNull(records);
        assertTrue(records.size() == 0);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsAddSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        Supplier mockSupplier = buildMockSupplier();
        
        handler.addSupplier(null, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedAddSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Supplier mockSupplier = buildMockSupplier();
        
        handler.addSupplier(userCredentialsMock, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullAddSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.addSupplier(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullSupplierUpdateSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
       
       handler.updateSupplier(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsUpdateSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
          
        Supplier mockSupplier = buildMockSupplier();
        
        handler.updateSupplier(null, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedUpdateSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Supplier mockSupplier = buildMockSupplier();
        
        handler.updateSupplier(userCredentialsMock, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullUpdateSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.updateSupplier(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
                
        Supplier mockSupplier = buildMockSupplier();
        
        handler.isDeleteable(null, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
        
        Supplier mockSupplier = buildMockSupplier();
        
        handler.isDeleteable(userCredentialsMock, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullIsDeleteableSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test
    public void testIsDeleteableSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        Supplier mockSupplier = buildMockSupplier();
        // Supplier 14 has no Products
        mockSupplier.setId(14);
         
        assertTrue(handler.isDeleteable(userCredentialsMock, mockSupplier));
    }
    
    @Test
    public void testIsNotDeleteableSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        Supplier mockSupplier = buildMockSupplier();
        // Supplier 12 has Products
        mockSupplier.setId(12);
        
        assertFalse(handler.isDeleteable(userCredentialsMock, mockSupplier));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsIsDeleteableId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        handler.isDeleteable(null, 21);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedIsDeleteableId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(false);
              
        handler.isDeleteable(userCredentialsMock, 21);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullIsDeleteableId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
             
        handler.isDeleteable(userCredentialsMock, null);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testIsDeleteableId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        // Supplier 12 has no Products
        assertTrue(handler.isDeleteable(userCredentialsMock, 12));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testIsNotDeleteableId() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
                  
        assertFalse(handler.isDeleteable(userCredentialsMock, 14));
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullCredentialsDeleteSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
               
        Supplier mockSupplier = buildMockSupplier();
        
        handler.deleteEntity(null, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testUnauthorizedDeleteSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(false);
        
        Supplier mockSupplier = buildMockSupplier();
        
        handler.deleteEntity(userCredentialsMock, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testNullDeleteSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        handler.deleteEntity(userCredentialsMock, null);
    }
    
    @Test(expected = NonDeleteableRecordException.class)  
    public void testNotDeleteableSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Supplier mockSupplier = buildMockSupplier();
        // Supplier 12 has Products
        mockSupplier.setId(12);
        
        handler.deleteEntity(userCredentialsMock, mockSupplier);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidEntityDeleteableProduct() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        CustomerEntity mockCustomerEntity = new CustomerEntity();
         
        handler.deleteEntity(userCredentialsMock, mockCustomerEntity);
    }
    
    @Test(expected = IllegalArgumentException.class)  
    public void testInvalidEntityIsDeleteableCustomer() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        CustomerEntity mockCustomerEntity = new CustomerEntity();
         
        assertTrue(handler.isDeleteable(userCredentialsMock, mockCustomerEntity));
    }
   
    private Supplier buildMockSupplier()
    {
        Supplier mockSupplier = new Supplier();
        // leave the id as default, the database will generate it
        mockSupplier.setCompanyName("ABC Inc.");
        mockSupplier.setContactName("John Smith");
        Location mockLocation = new Location();
        mockLocation.setCity("Dawson City");
        mockLocation.setCountry("Canada");
        mockSupplier.setLocation(mockLocation);
        Phone mockPhone = new Phone();
        mockPhone.setNumber("(867)993-5566");
        mockSupplier.setPhone(mockPhone);
        
        return mockSupplier;
    }
}
