package org.example.customerdao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.when;

import java.io.File;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.config.Configurator;
import org.dbunit.Assertion;
import org.dbunit.dataset.ITable;
import org.example.customer.Supplier;
import org.example.customer.utility.Location;
import org.example.customer.utility.Phone;
import org.example.customerdao.testcategories.SmokeTest;
import org.example.dao.testUtils.jdbc.DBUnitJDBCUtility;
import org.example.websecurity.UserCredentials;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;

public class SupplierDAOSlowTests
{
    private static String configDir = "src" + File.separator
            + "test" + File.separator + "resources" + File.separator + "data";
    
    private static final String SCHEMA_FILE = configDir + File.separator + "Customer.sql";
    private static final String DATA_FILE = configDir + File.separator + "fullDB.xml";
    private static final String SUPPLIER_ADD_FILE = configDir + File.separator + "addSupplier.xml";
    private static final String SUPPLIER_UPDATE_FILE = configDir + File.separator + "updateSupplier.xml";
    private static final String SUPPLIER_DELETE_FILE = configDir + File.separator + "deleteSupplier.xml";
    
    private static DBUnitJDBCUtility utility = null;
    
    @Before  // (re)build the Database before each test
    public void setup()
    {
        Configurator.setLevel(LogManager.getLogger(Supplier.class).getName(), Level.WARN);
        
        try
        {
            utility = new DBUnitJDBCUtility(SCHEMA_FILE, DATA_FILE);
        }
        catch (Exception e)
        {
            fail(e.getLocalizedMessage());
        }
    }  
    
    @After
    public void shutdown()
    {
        if (utility != null)
        {
            utility.shutdown();
        }
    }
    
    @Category(SmokeTest.class)
    @Test(timeout = 3000) // 3 sec (3000) millisec timeout
    public void testAddSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Supplier mockSupplier = buildMockSupplier();
        
        int expected = 30;
        int actual = handler.addSupplier(userCredentialsMock, mockSupplier);
        assertEquals(expected, actual);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("SUPPLIER");
            ITable expectedTable = utility.getTableFromFile(SUPPLIER_ADD_FILE, "SUPPLIER");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
    
    @Test(timeout = 3000) 
    public void testUpdateSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
        
        Supplier mockSupplier3 = handler.findSupplierById(userCredentialsMock, 3);
        mockSupplier3.getLocation().setCity("Detroit");
        
        handler.updateSupplier(userCredentialsMock, mockSupplier3);
        
        Supplier mockSupplier14 = handler.findSupplierById(userCredentialsMock, 14);
        mockSupplier14.getFax().setNumber("(0544) 60604");
        
        handler.updateSupplier(userCredentialsMock, mockSupplier14);
        
        Supplier mockSupplier25 = handler.findSupplierById(userCredentialsMock, 25);
        mockSupplier25.setCompanyName("Ma Maison - Montréal");
        
        handler.updateSupplier(userCredentialsMock, mockSupplier25);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("SUPPLIER");
            ITable expectedTable = utility.getTableFromFile(SUPPLIER_UPDATE_FILE, "SUPPLIER");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }
      
    @Test(timeout = 1000) // 1 sec (1000) millisec timeout
    public void testDeleteSupplier() 
            throws Exception
    {
        SupplierDAO handler = new SupplierDAOImpl();
        handler.setReadWriteDS(utility.getDataSource());
        handler.setReadOnlyDS(utility.getDataSource());
        
        UserCredentials userCredentialsMock = Mockito.mock(UserCredentials.class);
        when(userCredentialsMock.hasRole("worker")).thenReturn(true);
        when(userCredentialsMock.hasRole("manager")).thenReturn(true);
             
        Supplier mockSupplier = buildMockSupplier();
        // Supplier 14 has no Products
        mockSupplier.setId(14);
         
        handler.deleteEntity(userCredentialsMock, mockSupplier);
        
        try
        {
            // Fetch database data after executing your code
            ITable actualTable = utility.getTableFromDatabase("SUPPLIER");
            ITable expectedTable = utility.getTableFromFile(SUPPLIER_DELETE_FILE, "SUPPLIER");

            // Assert actual database table match expected table
            // This will check every row, and every column of the table
            Assertion.assertEquals(expectedTable, actualTable);
        }
        catch (Exception e1)
        {
            fail(e1.getLocalizedMessage());
        }
    }

    private Supplier buildMockSupplier()
    {   
        Supplier mockSupplier = new Supplier();
        // leave the id as default, the database will generate it
        mockSupplier.setCompanyName("Service Unlimited");
        mockSupplier.setContactName("George Robers Jr.");
        Location mockLocation = new Location();
        mockLocation.setCity("Memphis");
        mockLocation.setCountry("USA");
        mockSupplier.setLocation(mockLocation);
        Phone mockPhone = new Phone();
        mockPhone.setNumber("(389) 555-1212");
        mockSupplier.setPhone(mockPhone);
        
        return mockSupplier;
    }
}
